<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\Recovery;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Tools\OTPMailer;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\User\UserRecoveryTable;
use Erlage\Photogram\Data\Models\User\Recovery\UserRecoveryEnum;
use Erlage\Photogram\Data\Models\User\Recovery\UserRecoveryModel;
use Erlage\Photogram\Data\Models\User\Recovery\UserRecoveryFinder;
use Erlage\Photogram\Data\Models\User\Recovery\UserRecoveryBuilder;

final class UserRecoveryActions extends ExceptionalRequests
{
    public static function start(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(UserTable::ID, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | whether generate a new recovery reqeust or reuse previous one(if exists)
            |--------------------------------------------------------------------------
            */

            $generateNewRequest = true;

            /*
            |--------------------------------------------------------------------------
            | find the most recent recovery request
            |--------------------------------------------------------------------------
            */

            $userRecoverFinder = (new UserRecoveryFinder())
                -> setUserId($targetUserModel -> getId())
                -> setMetaIsExpired(UserRecoveryEnum::META_IS_EXPIRED_NO)
                -> limit('1')
                -> find();

            /*
            |--------------------------------------------------------------------------
            | if found make sure not expired
            |--------------------------------------------------------------------------
            */

            if ($userRecoverFinder -> isFound())
            {
                $userRecoveryModel = $userRecoverFinder -> popModelFromResults();

                if ($userRecoveryModel -> isObsolute())
                {
                    $userRecoveryModel -> update(
                        array(
                            UserRecoveryTable::META_IS_EXPIRED => UserRecoveryEnum::META_IS_EXPIRED_YES,
                        )
                    );

                    $userRecoveryModel -> save();
                }
                else
                {
                    $generateNewRequest = false;
                }
            }

            /*
            |--------------------------------------------------------------------------
            | finally generate new request if not found or expired
            |--------------------------------------------------------------------------
            */

            if ($generateNewRequest)
            {
                $userRecoveryModel = (new UserRecoveryBuilder())
                    -> setUserId($targetUserModel -> getId())
                    -> dispense();

                $userRecoveryModel -> save();
            }

            /*
            |--------------------------------------------------------------------------
            | send email to user's email
            |--------------------------------------------------------------------------
            */

            (new OTPMailer())
                -> setOTP($userRecoveryModel -> getMetaAccessOTP())
                -> setUserModel($targetUserModel)
                -> dispatchRecoveryEmail();

            /*
            |--------------------------------------------------------------------------
            | create a dummy model containing only reference id(primary id) of request
            | which they can use while submitting otp in confirmation step
            |--------------------------------------------------------------------------
            */

            $refOnlyUserRecoveryModel = (new UserRecoveryBuilder())
                -> setUserId('_')
                -> setMetaAccessOTP('_')
                -> setMetaAccessToken('_')
                -> setStampLastUpdate('_')
                -> setStampRegistration('_')
                -> dispense();

            // add id of recovery entry

            $refOnlyUserRecoveryModel -> update(
                array(
                    UserRecoveryTable::ID      => $userRecoveryModel -> getId(),
                    UserRecoveryTable::USER_ID => $targetUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | add dummy model to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserRecoveryTable::getTableName(), $refOnlyUserRecoveryModel -> getDataMap());
        });
    }

    public static function confirm(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userRecoveryIdFromReq = self::$request -> findKey(
                UserRecoveryTable::ID,
                RequestTable::PAYLOAD,
                UserRecoveryTable::TABLE_NAME
            );

            $metaAccessOTPFromReq = self::$request -> findKey(
                UserRecoveryTable::META_ACCESS_OTP,
                RequestTable::PAYLOAD,
                UserRecoveryTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userRecoveryIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target recovery request exists
            |--------------------------------------------------------------------------
            */

            $targetUserRecoveryModel = UserRecoveryModel::findFromId_throwException($userRecoveryIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            /**
             * flag: not used anywhere else
             */
            $targetUserModel = UserModel::findFromId_throwException($targetUserRecoveryModel -> getUserId());

            /*
            |--------------------------------------------------------------------------
            | check recovery request expiry
            |--------------------------------------------------------------------------
            */

            if ($targetUserRecoveryModel -> isExpired())
            {
                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            if ($targetUserRecoveryModel -> isObsolute())
            {
                $targetUserRecoveryModel -> update(
                    array(
                        UserRecoveryTable::META_IS_EXPIRED => UserRecoveryEnum::META_IS_EXPIRED_YES,
                    )
                );

                $targetUserRecoveryModel -> save();

                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            /*
            |--------------------------------------------------------------------------
            | confirm otp
            |--------------------------------------------------------------------------
            */

            if ($targetUserRecoveryModel -> getMetaAccessOTP() != $metaAccessOTPFromReq)
            {
                /*
                |--------------------------------------------------------------------------
                | increment attempt count and expire is it maxed out
                |--------------------------------------------------------------------------
                */

                $timesFailed = (string) (
                    (
                        (int) ($targetUserRecoveryModel -> getMetaTimesFailed())
                    ) + 1
                );

                $fieldsToUpdate = array(
                    UserRecoveryTable::META_TIMES_FAILED => $timesFailed,
                );

                if ((int) $timesFailed >= Settings::getInt(ServerConstants::SS_INT_OTP_LIFETIME_IN_TRIES))
                {
                    $fieldsToUpdate[UserRecoveryTable::META_IS_EXPIRED] = UserRecoveryEnum::META_IS_EXPIRED_YES;
                }

                $targetUserRecoveryModel -> update($fieldsToUpdate);

                $targetUserRecoveryModel -> save();

                /*
                |--------------------------------------------------------------------------
                | stop and tell client about incorrect otp
                |--------------------------------------------------------------------------
                */

                throw new RequestException(ResponseConstants::D_ERROR_OTP_MISMATCH);
            }

            /*
            |--------------------------------------------------------------------------
            | else send user model which contains everything including access token
            | which user can use in reset password request
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserRecoveryTable::getTableName(), $targetUserRecoveryModel -> getDataMap());
        });
    }

    public static function resetPassword(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userRecoveryIdFromReq = self::$request -> findKey(
                UserRecoveryTable::ID,
                RequestTable::PAYLOAD,
                UserRecoveryTable::TABLE_NAME
            );

            $metaAccessTokenFromReq = self::$request -> findKey(
                UserRecoveryTable::META_ACCESS_TOKEN,
                RequestTable::PAYLOAD,
                UserRecoveryTable::TABLE_NAME
            );

            $newPasswordFromReq = self::$request -> findKey(
                UserTable::EXTRA_NEW_PASSWORD,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            $retypeNewPasswordFromReq = self::$request -> findKey(
                UserTable::EXTRA_RETYPE_NEW_PASSWORD,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            self::ensureValue(
                ResponseConstants::ERROR_BAD_REQUEST_MSG,
                $userRecoveryIdFromReq,
                $metaAccessTokenFromReq,
                $newPasswordFromReq,
                $retypeNewPasswordFromReq
            );

            /*
            |--------------------------------------------------------------------------
            | ensure target recovery request exists
            |--------------------------------------------------------------------------
            */

            $targetUserRecoveryModel = UserRecoveryModel::findFromId_throwException($userRecoveryIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            /**
             * flag: not used anywhere else
             */
            $targetUserModel = UserModel::findFromId_throwException($targetUserRecoveryModel -> getUserId());

            /*
            |--------------------------------------------------------------------------
            | check recovery request expiry
            |--------------------------------------------------------------------------
            */

            if ($targetUserRecoveryModel -> isExpired())
            {
                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            if ($targetUserRecoveryModel -> isObsolute())
            {
                $targetUserRecoveryModel -> update(
                    array(
                        UserRecoveryTable::META_IS_EXPIRED => UserRecoveryEnum::META_IS_EXPIRED_YES,
                    )
                );

                $targetUserRecoveryModel -> save();

                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            /*
            |--------------------------------------------------------------------------
            | confirm access token
            |--------------------------------------------------------------------------
            */

            if ($targetUserRecoveryModel -> getMetaAccessToken() != $metaAccessTokenFromReq)
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | check whether new & retype password are different
            |--------------------------------------------------------------------------
            */

            if ($newPasswordFromReq !== $retypeNewPasswordFromReq)
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_PASSWORD_MISMATCH_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | login as current user
            |--------------------------------------------------------------------------
            */

            self::$userSession -> loginAs($targetUserModel);

            self::userAuthenticate();

            /*
            |--------------------------------------------------------------------------
            | update user's fields
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> update(
                array(
                    UserTable::PASSWORD    => $newPasswordFromReq,
                    UserTable::META_ACCESS => '',
                )
            );

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | expire the recovery request
            |--------------------------------------------------------------------------
            */

            $targetUserRecoveryModel -> update(
                array(
                    UserRecoveryTable::META_IS_EXPIRED => UserRecoveryEnum::META_IS_EXPIRED_YES,
                )
            );

            $targetUserRecoveryModel -> save();

            /*
            |--------------------------------------------------------------------------
            | send current user
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
        });
    }
}
